'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _assign = require('babel-runtime/core-js/object/assign');

var _assign2 = _interopRequireDefault(_assign);

var _synchronizationListener = require('../clients/metaApi/synchronizationListener');

var _synchronizationListener2 = _interopRequireDefault(_synchronizationListener);

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Exposes MetaApi MetaTrader API connection to consumers
 */
class MetaApiConnection extends _synchronizationListener2.default {

  /**
   * Constructs MetaApi MetaTrader Api connection
   * @param {MetaApiWebsocketClient} websocketClient MetaApi websocket client
   * @param {MetatraderAccount} account MetaTrader account id to connect to
   * @param {String} [application] application to use
   */
  constructor(websocketClient, account, application) {
    super();
    this._websocketClient = websocketClient;
    this._account = account;
    this._logger = _logger2.default.getLogger('MetaApiConnection');
    this._application = application;
  }

  /**
   * Common trade options
   * @typedef {Object} TradeOptions
   * @property {String} [comment] optional order comment. The sum of the line lengths of the comment and the
   * clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {String} [clientId] optional client-assigned id. The id value can be assigned when submitting a trade and
   * will be present on position, history orders and history deals related to the trade. You can use this field to bind
   * your trades to objects in your application and then track trade progress. The sum of the line lengths of the
   * comment and the clientId must be less than or equal to 26. For more information see
   * https://metaapi.cloud/docs/client/clientIdUsage/
   * @property {Number} [magic] optional magic (expert id) number. If not set default value specified in account entity
   * will be used.
   * @property {Number} [slippage] optional slippage in points. Should be greater or equal to zero. In not set,
   * default value specified in account entity will be used. Slippage is ignored if execution mode set to
   * SYMBOL_TRADE_EXECUTION_MARKET in symbol specification. Not used for close by orders.
   */

  /**
   * Market trade options
   * @typedef {TradeOptions} MarketTradeOptions
   * @property {Array<String>} [fillingModes] optional allowed filling modes in the order of priority. Default is to
   * allow all filling modes and prefer ORDER_FILLING_FOK over ORDER_FILLING_IOC. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_type_filling for extra
   * explanation
   */

  /**
   * Market trade options
   * @typedef {MarketTradeOptions} CreateMarketTradeOptions
   * @property {TrailingStopLoss} [trailingStopLoss] distance trailing stop loss configuration
   * @property {String} [stopPriceBase] defines the base price to calculate SL/TP relative to for pending order
   * requests. Default is CURRENT_PRICE, one of CURRENT_PRICE
   */

  /**
   * Pending order trade options
   * @typedef {TradeOptions} PendingTradeOptions
   * @property {ExpirationOptions} [expiration] optional pending order expiration settings. See Pending order expiration
   * settings section
   * @property {TrailingStopLoss} [trailingStopLoss] distance trailing stop loss configuration
   * @property {String} [stopPriceBase] defined the base price to calculate SL/TP relative to for *_MODIFY and pending
   * order requests. STOP_PRICE means the SL/TP is relative to previous SL/TP value. Default is OPEN_PRICE, one of
   * CURRENT_PRICE, OPEN_PRICE
   * @property {String} [openPriceUnits] open price units. ABSOLUTE_PRICE means the that the value of openPrice field
   * is a final open price value. RELATIVE* means that the openPrice field value contains relative open price expressed
   * either in price, points, pips, account currency or balance percentage. Default is ABSOLUTE_PRICE. One of
   * ABSOLUTE_PRICE, RELATIVE_PRICE, RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY, RELATIVE_BALANCE_PERCENTAGE
   */

  /**
   * Options for creating a stop limit pending order
   * @typedef {PendingTradeOptions} StopLimitPendingTradeOptions
   * @property {String} [openPriceBase] defined the base price to calculate open price relative to for ORDER_MODIFY
   * and pending order requests. Default is CURRENT_PRICE for pending orders or STOP_LIMIT_PRICE for stop limit orders.
   * One of CURRENT_PRICE, OPEN_PRICE, STOP_LIMIT_PRICE
   * @property {String} [stopLimitPriceUnits] stop limit price units. ABSOLUTE_PRICE means the that the value of
   * stopLimitPrice field is a final stop limit price value. RELATIVE* means that the stopLimitPrice field value
   * contains relative stop limit price expressed either in price, points, pips, account currency or balance percentage.
   * Default is ABSOLUTE_PRICE. One of ABSOLUTE_PRICE, RELATIVE_PRICE, RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY,
   * RELATIVE_BALANCE_PERCENTAGE
   */

  /**
   * Options for modifying orders
   * @typedef {Object} ModifyOrderOptions
   * @property {TrailingStopLoss} [trailingStopLoss] distance trailing stop loss configuration
   * @property {String} [stopPriceBase] defined the base price to calculate SL/TP relative to for *_MODIFY and pending
   * order requests. STOP_PRICE means the SL/TP is relative to previous SL/TP value. Default is OPEN_PRICE, one of
   * CURRENT_PRICE, OPEN_PRICE, STOP_PRICE
   * @property {String} [openPriceUnits] open price units. ABSOLUTE_PRICE means the that the value of openPrice field
   * is a final open price value. RELATIVE* means that the openPrice field value contains relative open price expressed
   * either in price, points, pips, account currency or balance percentage. Default is ABSOLUTE_PRICE. One of
   * ABSOLUTE_PRICE, RELATIVE_PRICE, RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY, RELATIVE_BALANCE_PERCENTAGE
   * @property {String} [openPriceBase] defined the base price to calculate open price relative to for ORDER_MODIFY
   * and pending order requests. Default is CURRENT_PRICE for pending orders or STOP_LIMIT_PRICE for stop limit orders.
   * One of CURRENT_PRICE, OPEN_PRICE, STOP_LIMIT_PRICE
   * @property {String} [stopLimitPriceUnits] stop limit price units. ABSOLUTE_PRICE means the that the value of
   * stopLimitPrice field is a final stop limit price value. RELATIVE* means that the stopLimitPrice field value
   * contains relative stop limit price expressed either in price, points, pips, account currency or balance percentage.
   * Default is ABSOLUTE_PRICE. One of ABSOLUTE_PRICE, RELATIVE_PRICE, RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY,
   * RELATIVE_BALANCE_PERCENTAGE
   * @property {String} [stopLimitPriceBase] Defined the base price to calculate stop limit price relative to for
   * ORDER_MODIFY requests. One of CURRENT_PRICE, STOP_LIMIT_PRICE
   */

  /**
   * Pending order expiration settings
   * @typedef {Object} ExpirationOptions
   * @property {String} type pending order expiration type. See
   * https://www.mql5.com/en/docs/constants/tradingconstants/orderproperties#enum_order_type_time for the list of
   * possible options. MetaTrader4 platform supports only ORDER_TIME_SPECIFIED expiration type. One of ORDER_TIME_GTC,
   * ORDER_TIME_DAY, ORDER_TIME_SPECIFIED, ORDER_TIME_SPECIFIED_DAY
   * @property {Date} [time] optional pending order expiration time. Ignored if expiration type is not one of
   * ORDER_TIME_DAY or ORDER_TIME_SPECIFIED
   */

  /**
   * Stop options
   * @typedef {Object} StopOptions
   * @property {number} value stop (SL or TP) value
   * @property {string} units stop units. ABSOLUTE_PRICE means the that the value of value field is a final stop value.
   * RELATIVE_* means that the value field value contains relative stop expressed either in price, points, pips, account
   * currency or balance percentage. Default is ABSOLUTE_PRICE. Allowed values are ABSOLUTE_PRICE, RELATIVE_PRICE,
   * RELATIVE_POINTS, RELATIVE_PIPS, RELATIVE_CURRENCY, RELATIVE_BALANCE_PERCENTAGE
   */

  /**
   * Creates a market buy order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {CreateMarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createMarketBuyOrder(symbol, volume, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_BUY', symbol, volume }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a market sell order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {CreateMarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createMarketSellOrder(symbol, volume, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_SELL', symbol, volume }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a limit buy order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {String} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order limit price
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {PendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createLimitBuyOrder(symbol, volume, openPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_BUY_LIMIT', symbol,
      volume, openPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a limit sell order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order limit price
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {PendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createLimitSellOrder(symbol, volume, openPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_SELL_LIMIT', symbol,
      volume, openPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a stop buy order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order stop price
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {PendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createStopBuyOrder(symbol, volume, openPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_BUY_STOP', symbol,
      volume, openPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a stop sell order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order stop price
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {PendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createStopSellOrder(symbol, volume, openPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_SELL_STOP', symbol,
      volume, openPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a stop limit buy order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order stop price
   * @param {number} stopLimitPrice the limit order price for the stop limit order
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {StopLimitPendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createStopLimitBuyOrder(symbol, volume, openPrice, stopLimitPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_BUY_STOP_LIMIT',
      symbol, volume, openPrice, stopLimitPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Creates a stop limit sell order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {number} volume order volume
   * @param {number} openPrice order stop price
   * @param {number} stopLimitPrice the limit order price for the stop limit order
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {StopLimitPendingTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  createStopLimitSellOrder(symbol, volume, openPrice, stopLimitPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_TYPE_SELL_STOP_LIMIT',
      symbol, volume, openPrice, stopLimitPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Modifies a position (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} positionId position id to modify
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {TrailingStopLoss} trailingStopLoss distance trailing stop loss configuration
   * @param {String} [stopPriceBase] defines the base price to calculate SL relative to for POSITION_MODIFY and
   * pending order requests. Default is OPEN_PRICE. One of CURRENT_PRICE, OPEN_PRICE, STOP_PRICE
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  modifyPosition(positionId, stopLoss, takeProfit, trailingStopLoss, stopPriceBase) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'POSITION_MODIFY', positionId, trailingStopLoss, stopPriceBase }, this._generateStopOptions(stopLoss, takeProfit)), this._application);
  }

  /**
   * Partially closes a position (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} positionId position id to modify
   * @param {number} volume volume to close
   * @param {MarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  closePositionPartially(positionId, volume, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'POSITION_PARTIAL', positionId,
      volume }, options || {}), this._application);
  }

  /**
   * Fully closes a position (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} positionId position id to modify
   * @param {MarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  closePosition(positionId, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'POSITION_CLOSE_ID', positionId }, options || {}), this._application);
  }

  /**
   * Fully closes a position (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} positionId position id to close by opposite position
   * @param {string} oppositePositionId opposite position id to close
   * @param {MarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  closeBy(positionId, oppositePositionId, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'POSITION_CLOSE_BY', positionId,
      closeByPositionId: oppositePositionId }, options || {}), this._application);
  }

  /**
   * Closes positions by a symbol(see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} symbol symbol to trade
   * @param {MarketTradeOptions} options optional trade options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  closePositionsBySymbol(symbol, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'POSITIONS_CLOSE_SYMBOL', symbol }, options || {}), this._application);
  }

  /**
   * Modifies a pending order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} orderId order id (ticket number)
   * @param {number} openPrice order stop price
   * @param {number|StopOptions} [stopLoss] stop loss price
   * @param {number|StopOptions} [takeProfit] take profit price
   * @param {ModifyOrderOptions} [options] optional modify order options
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  modifyOrder(orderId, openPrice, stopLoss, takeProfit, options = {}) {
    return this._websocketClient.trade(this._account.id, (0, _assign2.default)({ actionType: 'ORDER_MODIFY', orderId, openPrice }, this._generateStopOptions(stopLoss, takeProfit), options || {}), this._application);
  }

  /**
   * Cancels order (see https://metaapi.cloud/docs/client/websocket/api/trade/).
   * @param {string} orderId order id (ticket number)
   * @returns {Promise<TradeResponse>} promise resolving with trade result
   * @throws {TradeError} on trade error, check error properties for error code details
   */
  cancelOrder(orderId) {
    return this._websocketClient.trade(this._account.id, { actionType: 'ORDER_CANCEL', orderId }, this._application);
  }

  /**
   * Returns MetaApi account
   * @return {MetatraderAccount} MetaApi account
   */
  get account() {
    return this._account;
  }
  _generateStopOptions(stopLoss, takeProfit) {
    let trade = {};
    if (typeof stopLoss === 'number') {
      trade.stopLoss = stopLoss;
    } else if (stopLoss) {
      trade.stopLoss = stopLoss.value;
      trade.stopLossUnits = stopLoss.units;
    }
    if (typeof takeProfit === 'number') {
      trade.takeProfit = takeProfit;
    } else if (takeProfit) {
      trade.takeProfit = takeProfit.value;
      trade.takeProfitUnits = takeProfit.units;
    }
    return trade;
  }

  async _ensureSynchronized(instanceIndex, key) {
    let state = this._getState(instanceIndex);
    if (state && !this._closed) {
      try {
        const synchronizationResult = await this.synchronize(instanceIndex);
        if (synchronizationResult) {
          state.synchronized = true;
          state.synchronizationRetryIntervalInSeconds = 1;
        }
      } catch (err) {
        this._logger.error('MetaApi websocket client for account ' + this._account.id + ':' + instanceIndex + ' failed to synchronize', err);
        if (state.shouldSynchronize === key) {
          setTimeout(this._ensureSynchronized.bind(this, instanceIndex, key), state.synchronizationRetryIntervalInSeconds * 1000);
          state.synchronizationRetryIntervalInSeconds = Math.min(state.synchronizationRetryIntervalInSeconds * 2, 300);
        }
      }
    }
  }

  _getState(instanceIndex) {
    if (!this._stateByInstanceIndex['' + instanceIndex]) {
      this._stateByInstanceIndex['' + instanceIndex] = {
        instanceIndex,
        ordersSynchronized: {},
        dealsSynchronized: {},
        shouldSynchronize: undefined,
        synchronizationRetryIntervalInSeconds: 1,
        synchronized: false,
        lastDisconnectedSynchronizationId: undefined,
        lastSynchronizationId: undefined,
        disconnected: false
      };
    }
    return this._stateByInstanceIndex['' + instanceIndex];
  }

}
exports.default = MetaApiConnection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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