'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud historical market data API client
 */
class HistoricalMarketDataClient extends _metaApi2.default {

  /**
   * Constructs historical market data API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */
  constructor(httpClient, token, domain = 'agiliumtrade.agiliumtrade.ai') {
    super(httpClient, token, domain);
    this._host = `https://mt-market-data-client-api-v1.${domain}`;
  }

  /**
   * Returns historical candles for a specific symbol and timeframe from a MetaTrader account.
   * See https://metaapi.cloud/docs/client/restApi/api/retrieveMarketData/readHistoricalCandles/
   * @param {string} accountId MetaTrader account id
   * @param {string} symbol symbol to retrieve candles for (e.g. a currency pair or an index)
   * @param {string} timeframe defines the timeframe according to which the candles must be generated. Allowed values
   * for MT5 are 1m, 2m, 3m, 4m, 5m, 6m, 10m, 12m, 15m, 20m, 30m, 1h, 2h, 3h, 4h, 6h, 8h, 12h, 1d, 1w, 1mn. Allowed
   * values for MT4 are 1m, 5m, 15m 30m, 1h, 4h, 1d, 1w, 1mn
   * @param {Date} [startTime] time to start loading candles from. Note that candles are loaded in backwards direction, so
   * this should be the latest time. Leave empty to request latest candles.
   * @param {number} [limit] maximum number of candles to retrieve. Must be less or equal to 1000
   * @return {Promise<Array<MetatraderCandle>>} promise resolving with historical candles downloaded
   */
  async getHistoricalCandles(accountId, symbol, timeframe, startTime, limit) {
    symbol = encodeURIComponent(symbol);
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/historical-market-data/symbols/${symbol}/` + `timeframes/${timeframe}/candles`,
      method: 'GET',
      qs: {
        startTime,
        limit
      },
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    let candles = await this._httpClient.request(opts);
    candles = candles || [];
    candles.forEach(c => c.time = new Date(c.time));
    return candles;
  }

  /**
   * Returns historical ticks for a specific symbol from a MetaTrader account. This API is not supported by MT4
   * accounts.
   * See https://metaapi.cloud/docs/client/restApi/api/retrieveMarketData/readHistoricalTicks/
   * @param {string} accountId MetaTrader account id
   * @param {string} symbol symbol to retrieve ticks for (e.g. a currency pair or an index)
   * @param {Date} [startTime] time to start loading ticks from. Note that candles are loaded in forward direction, so
   * this should be the earliest time. Leave empty to request latest candles.
   * @param {number} [offset] number of ticks to skip (you can use it to avoid requesting ticks from previous request
   * twice)
   * @param {number} [limit] maximum number of ticks to retrieve. Must be less or equal to 1000
   * @return {Promise<Array<MetatraderTick>>} promise resolving with historical ticks downloaded
   */
  async getHistoricalTicks(accountId, symbol, startTime, offset, limit) {
    symbol = encodeURIComponent(symbol);
    const opts = {
      url: `${this._host}/users/current/accounts/${accountId}/historical-market-data/symbols/${symbol}/ticks`,
      method: 'GET',
      qs: {
        startTime,
        offset,
        limit
      },
      headers: {
        'auth-token': this._token
      },
      json: true
    };
    let ticks = await this._httpClient.request(opts);
    ticks = ticks || [];
    ticks.forEach(t => t.time = new Date(t.time));
    return ticks;
  }

}
exports.default = HistoricalMarketDataClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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