'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _metaApi = require('../metaApi.client');

var _metaApi2 = _interopRequireDefault(_metaApi);

var _randomstring = require('randomstring');

var _randomstring2 = _interopRequireDefault(_randomstring);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * metaapi.cloud CopyFactory history API (trade copying history API) client (see
 * https://metaapi.cloud/docs/copyfactory/)
 */
class HistoryClient extends _metaApi2.default {

  /**
   * Constructs CopyFactory history API client instance
   * @param {HttpClient} httpClient HTTP client
   * @param {String} token authorization token
   * @param {String} domain domain to connect to, default is agiliumtrade.agiliumtrade.ai
   */
  constructor(httpClient, token, domain = 'agiliumtrade.agiliumtrade.ai') {
    super(httpClient, token, domain);
    this._host = `https://copyfactory-application-history-master-v1.${domain}`;
  }

  /**
   * CopyFactory provider or subscriber user
   * @typedef {Object} CopyFactorySubscriberOrProviderUser
   * @property {String} id profile id
   * @property {String} name user name
   * @property {Array<CopyFactoryStrategyIdAndName>} strategies array of strategy IDs provided by provider
   * or subscribed to by subscriber
   */

  /**
   * CopyFactory strategy id and name
   * @typedef {Object} CopyFactoryStrategyIdAndName
   * @property {String} id unique strategy id
   * @property {String} name human-readable strategy name
   */

  /**
   * CopyFactory transaction
   * @typedef {Object} CopyFactoryTransaction
   * @property {String} id transaction id
   * @property {String} type transaction type (one of DEAL_TYPE_BUY, DEAL_TYPE_SELL, DEAL_TYPE_BALANCE,
   * DEAL_TYPE_CREDIT, DEAL_TYPE_CHARGE, DEAL_TYPE_CORRECTION, DEAL_TYPE_BONUS, DEAL_TYPE_COMMISSION,
   * DEAL_TYPE_COMMISSION_DAILY, DEAL_TYPE_COMMISSION_MONTHLY, DEAL_TYPE_COMMISSION_AGENT_DAILY,
   * DEAL_TYPE_COMMISSION_AGENT_MONTHLY, DEAL_TYPE_INTEREST, DEAL_TYPE_BUY_CANCELED, DEAL_TYPE_SELL_CANCELED,
   * DEAL_DIVIDEND, DEAL_DIVIDEND_FRANKED, DEAL_TAX). See
   * https://www.mql5.com/en/docs/constants/tradingconstants/dealproperties#enum_deal_type
   * @property {Date} time transaction time
   * @property {String} subscriberId CopyFactory subscriber id
   * @property {String} [symbol] symbol traded
   * @property {CopyFactorySubscriberOrProviderUser} subscriberUser strategy subscriber
   * @property {Boolean} demo demo account flag
   * @property {CopyFactorySubscriberOrProviderUser} providerUser strategy provider
   * @property {CopyFactoryStrategyIdAndName} strategy strategy
   * @property {String} [positionId] source position id
   * @property {String} [slavePositionId] slave position id
   * @property {Number} improvement high-water mark strategy balance improvement
   * @property {Number} providerCommission provider commission
   * @property {Number} platformCommission platform commission
   * @property {Number} [incomingProviderCommission] commission paid by provider to underlying providers
   * @property {Number} [incomingPlatformCommission] platform commission paid by provider to underlying providers
   * @property {Number} [quantity] trade volume
   * @property {Number} [lotPrice] trade lot price
   * @property {Number} [tickPrice] trade tick price
   * @property {Number} [amount] trade amount
   * @property {Number} [commission] trade commission
   * @property {Number} swap trade swap
   * @property {Number} profit trade profit
   * @property {CopyFactoryTransactionMetrics} [metrics] trade copying metrics such as slippage and latencies. Measured
   * selectively for copied trades
   */

  /**
   * Trade copying metrics such as slippage and latencies
   * @typedef {Object} CopyFactoryTransactionMetrics
   * @property {Number} [tradeCopyingLatency] trade copying latency, measured in milliseconds based on transaction time
   * provided by broker
   * @property {Number} [tradeCopyingSlippageInBasisPoints] trade copying slippage, measured in basis points (0.01
   * percent) based on transaction price provided by broker
   * @property {Number} [tradeCopyingSlippageInAccountCurrency] trade copying slippage, measured in account currency
   * based on transaction price provided by broker
   * @property {Number} [mtAndBrokerSignalLatency] trade signal latency introduced by broker and MT platform, measured
   * in milliseconds
   * @property {Number} [tradeAlgorithmLatency] trade algorithm latency introduced by CopyFactory servers, measured in
   * milliseconds
   * @property {Number} [mtAndBrokerTradeLatency] trade latency for a copied trade introduced by broker and MT platform,
   * measured in milliseconds
   */

  /**
   * Returns list of transactions on the strategies the current user provides to other users
   * https://metaapi.cloud/docs/copyfactory/restApi/api/history/getProvidedTransactions/
   * @param {Date} from time to load transactions from
   * @param {Date} till time to load transactions till
   * @param {Array<string>} [strategyIds] list of strategy ids to filter transactions by
   * @param {Array<string>} [subscriberIds] the list of CopyFactory subscriber account ids to filter by
   * @param {number} [offset] pagination offset. Default value is 0
   * @param {number} [limit] pagination limit. Default value is 1000
   * @return {Promise<Array<CopyFactoryTransaction>>} promise resolving with transactions found
   */
  async getProvidedTransactions(from, till, strategyIds, subscriberIds, offset, limit) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getProvidedTransactions');
    }
    let qs = {
      from,
      till
    };
    if (strategyIds) {
      qs.strategyId = strategyIds;
    }
    if (subscriberIds) {
      qs.subscriberId = subscriberIds;
    }
    if (offset !== undefined) {
      qs.offset = offset;
    }
    if (limit) {
      qs.limit = limit;
    }
    const opts = {
      url: `${this._host}/users/current/provided-transactions`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      qs,
      json: true
    };
    let transactions = await this._httpClient.request(opts);
    transactions.forEach(t => t.time = new Date(t.time));
    return transactions;
  }

  /**
   * Returns list of trades on the strategies the current user subscribed to
   * https://metaapi.cloud/docs/copyfactory/restApi/api/history/getSubscriptionTransactions/
   * @param {Date} from time to load transactions from
   * @param {Date} till time to load transactions till
   * @param {Array<String>} [strategyIds] list of strategy ids to filter transactions by
   * @param {Array<string>} [subscriberIds] the list of CopyFactory subscriber account ids to filter by
   * @param {Number} offset pagination offset. Default value is 0
   * @param {Number} limit pagination limit. Default value is 1000
   * @return {Promise<Array<CopyFactoryTransaction>>} promise resolving with transactions found
   */
  async getSubscriptionTransactions(from, till, strategyIds, subscriberIds, offset, limit) {
    if (this._isNotJwtToken()) {
      return this._handleNoAccessError('getSubscriptionTransactions');
    }
    let qs = {
      from,
      till
    };
    if (strategyIds) {
      qs.strategyId = strategyIds;
    }
    if (subscriberIds) {
      qs.subscriberId = subscriberIds;
    }
    if (offset !== undefined) {
      qs.offset = offset;
    }
    if (limit) {
      qs.limit = limit;
    }
    const opts = {
      url: `${this._host}/users/current/subscription-transactions`,
      method: 'GET',
      headers: {
        'auth-token': this._token
      },
      qs,
      json: true
    };
    let transactions = await this._httpClient.request(opts);
    transactions.forEach(t => t.time = new Date(t.time));
    return transactions;
  }

}
exports.default = HistoryClient;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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